/********************************************************************

 main.c
 
 Universal USB Data Logger, by Mauro Grassi, 2010.
 
 Began with generic USB connected libusb device using an 18F26J50 by Mauro Grassi, 
 Created ex Nihilo on August 24, 2010...
 Mature version uses an 18F27J53...

 Compilation Instructions
 
 Using the following full versions:
 
 mpasmwin.exe v5.37, mplink.exe v4.37, mcc18.exe v3.36, mplib.exe v4.37
 
 You must use the custom linker script provided, with no iz or i option 
 (iz would mean the POR functions would not work!)
 
 The compilation command line is as follows:
 
 -k --verbose -mL -Ls -pa=1
 
 - Disabled STVREN function
 - Extended Instruction Set Enabled
 - Optimizations All Enabled Procedural Abstraction Passes=1
 - Enable Integer Promotions NO
 - Treat Char as unsigned YES
 - Large Code Model
 - Large Data Model
 - Multi Bank Stack
 - Compile/Link In Project Directory

 Assembler Settings:
 
 - do not disable case sensitivity in the Assembler
 
 Version History:
 
 0.4:  USB, USART, and RTCC working...
 0.6:  integrated FAT file system and memory card system..
 0.7:  modified ff.c code for PIC18 architecture (const rom vs ram pointers)...
 0.8:  tested FF with read and write to file: OK... Added DMA support for memory card using the 18F26J50's 
 	   SPI DMA module (hardware support)... Added EEPROM emulation library (must not disable case sensitivity in the PIC18 Assembler)...
 	   Added memory card power control through a high power digital IO pin... 
 0.9:  added date for FAT system from the RTCC (note use CREATE_ALWAYS in the f_open to write the date)...
 0.92: get_fattime checks if year>=1980...
 0.94: ...
 
 1.00: open/read/write/close file RPC via USB (Remote Procedure Calls)...
 1.02: one wire support started... PWM on SD power pin for "voltage doubler"...
 1.04: one wire support with LEGACY and OVERDRIVE speeds using single IO pin done...
 1.10: one wire support using UART or general IO pin, with STRONG PULL UP support (using a Maxim-Dallas DS18B20+ Temperature Sensor To Test with...)
 1.12: added I2C support (tested with an AD7414 sensor from Analog Devices)... Added timeout for one wire routines (non blocking)...
 
 1.20: added SPI support shared bus with the memory card...
 1.24: added ADC support...

Hardware Changes: changed PIC from 18F26J50 to 18F27J53 (they are pin for pin compatible but the latter has more program memory...)

 1.30: basic version adapted for the 18F27J53. While the two chips (18F26J50 and 18F27J53 are pin for pin compatible, the codes for the PPS (Peripheral Pin Select) are not!) 
 1.40: changed to large code model, and using SD registers now to go over the 64KB program memory limit of the previous PIC... capture module working...
 1.46: capture object system almost complete with 7 channels...
 1.48: ...
 1.50: ...
 1.60: ...
 1.70: changed to normal mode from extended mode (...)
 1.72: added openADCObject functions...
 1.80: working with PC host, frequency and ADC displays on PC host...
 1.98: added logExpressions support...
 1.99: grammar is flexible in the PC host, added evaluateEvaluationArray function...
 2.00: with command grammar...
 
 Hardware Changes below, changed to PCB pin outs...
 
 2.10: first PCB version (different pinouts)...
 2.20: added keys system using comparator 1 and timer 4...
 
 2.40: enlarged the stack beyond 256 bytes (it is just not enough for some of the complex FAT functions if using too much PA)... (must use -Ls option to the C18 compiler)...
 2.41: added dynamic duration objects...
 2.42: added logical channels infrastructure, made a union out of SPI DMA buffer and some temporary working space...
 2.44: ...
 2.46: ....
 
 Change to the circuit here, D4 and D5 labels swapped, and anode of (now) D5 is connected to the IO pin (LED1 control) to allow it to function as a doubler.
 
 2.50: new version with voltage doubler control circuit....
 2.60: led adapted to new circuit layout, multiplexed with voltage doubler circuit...
 2.62: ...
 2.64: serial USB Pipe improved to a FIFO...
 3.30: numbering consistent with PC host version... implementation of saveVM and restoreVM functions///
 
 Compilation Instructions changed...
 
 3.40: changed back to EXTENDED MODE for great size and speed savings (15% smaller code!) removed logExpressions support
 3.64: new global functions, I2C script support...
 3.70: improvements to the GPS module interface...
  
 3.80: improvements to the USB system, to avoid contentions...
 3.90: main improvements to the PC host software...
 
 3.96: expanded the data address space to an unsigned long and the actual data space is approx. 256KB
 	   changed linker to iz option... LONG_ADDRESSES support!

 3.99: added clearCache function, called when the VM is restored....

 4.00: extensive PC host simulation support with File IO, and changed the execution limit to 
 	   apply only on VM_RUN_MODE, so that if a HALT instruction is reached at the same time as
 	   an execution limit, the HALT takes precedence... added real time VM timer for VM machine...
 4.02: support for ALARMs for the RTCC...
 4.10: improvements add f_close before every open, and changes to the f_close function...
 	   added READ_ONLY caches (eg. program memory) which improves performance markedly...
 	   enabled the config bit STVREN... added CRC checking for restore settings...
 4.20: changes to the interrupt system for better performance...
	...
		
 4.50:	global inbuilt functions development infrastructure in place... 	    	            

 4.60:  new functions and virtual machine support for byte memory accesses, as well as string support...
 4.70:  better VME FSM...
 
 4.90:	...
 
 4.96:  added pow support, more global function support...
 5.00:  more global functions, improvements to the time subroutines...
 
 Changes to the Opcodes, possible code incompatibilities with earlier versions... 
 (OPCODE_NOP_HALT = 0x00, to prevent runaway executions on file read failures...)
 
 5.10:  huge performance improvements to the FAT filesystem, keys etc... CRC checking for VMs too...
 5.12:  ...
 
 5.20:  Optimizations for one VM VMEnvironment... Optimizations to the VM REAL TIME component...
 		VM REAL TIME support back to low priority interrupt...

 5.30: 	Start and Stop Meta Opcodes for the VMEnvironment, ie Process Control...
 5.40:  Memory Card Information is correct...
 5.50:  PC host improvements...
 5.52:  Integrated Capture module into VM globals...
 5.60:	Automatic Frequency Scaling for capture system...
 5.70:  Added Counters to the Capture System...
 5.72:	Added maths functions (random.c recycled from the DAB+ Tuner project!)
 
 5.80:  Log Output incorporated as a cache...
 5.90:  New globals... cardLED...
 
 5.94:  the popindb function was using the indirect addressing alignment of the general popind function, fixed!
 5.96:  improvements to the LED system...
 5.98:  sleep until command... time until command, time accumulator, etc... added LITI_ARG_TYPE for greater efficiency...
 6.00:  simple change to allow global functions to detect default arguments through the vm->temp field...

 6.10:	adds RAM to RAM DMA for better efficiency... SD card system DMA is direct now, saving RAM...
 
 6.12:  ...
 
 6.20:  pipes support... support for LFNs, up to 32 characters long... (properly fixed for the PIC18 micros...)
 
 6.24:  USE_PMDIS option for better power efficiency...
 
 6.30:	SLEEP modes...
 
 6.40:	added comparator 3 support for checking memory card eject and reconnect (interrupt driven)
 
 6.60:	new TIME_T definition...
 6.70:  ...
 6.80:  mutexes for PC host...
 6.90:  fixes the bulk transfers with fillPacket not returning the correct pointer for one direction...
 6.98:  internal global function calls are possible through the USB, added USB pipe...
 7.00a:	captive device with serial USB pipe...
 7.00b: added openPipes and closePipes opcodes and getPipes global function...
 7.10:  system logging cache...
 7.20a: hardware sharing infrastructure on a script basis...
 7.22:  ...
 7.24:	...
 7.30:  better hardware descriptor logic... added OPCODE_HEADER_OFFSET_WRITE...
 7.52:  global variables infrastructure with VM offsets...
 7.60:	fully integrated one wire module...
 7.64:	added NMEA support for print command and myultoabase commands for printing to other bases...
 7.70:  strings variable support...
 7.80:  improvements to the preludeBinary function so it is fully symmetric for non commutative binary operations...
 		added base() macro for print arguments with three arguments...
 7.82:  added SPI peripheral support... translate from logical to physical capture channels...
 7.84:	...
 7.90:  improvements to the PC host...
 7.94:  vm->typeMode reset on global function calls...
 7.96:  improvements...
 8.00a: many new improvements...
 8.01:	...
 8.10:  environment super task! syncSettings, many improvements...
 8.30:  if an SD card gives an error on init, memoryCardTask no longer retries continually, until it is
 		ejected...
 8.40:  banking optimizer turned off in the optimizer settings... deleteLog() function...
 8.50:  improvements... syncSettings() optimized and working... 
 8.60:	STVREN disabled...
 8.78:  ...  		      
 8.80:  improvements, including serial port receive functions...
 8.90:  ...
 9.00:  PC host compiler optimization enabled (copy propagation...)  
 9.02:  added DS LIMIT function for strings...
 9.10:
 9.12:
 9.20:  improvements, including file buffering!
 9.30:  no sleep if host is alive...
 9.32: 	...

 9.80:  many new improvements... new Global Variables system... added serial and nmea commands...
 9.82:  added the separate GPS nmea match string...
 9.84:  added nmea and matchnmea commands...
 9.88:  improved GPS with auto decoding of GPRMC sentences and also allows for the general match case...

 9.89:	REMOVED support for EEPROM emulation, to save program memory space (can no longer justify using so much
 		much program memory for so little gain...)

 9.90:  Bootloader fixes intermittent switching lag... Many other improvements to the PC host program...
		Much better serial bootloader, with synchronisation!
		Secondary bootloader (over UART) can update the USB bootloader, if necessary...
		Full check box functionality in PC host, also full help information in PC host...

	    Compiler & VM engine supports automatic variable sizing for the global variables...
		New globals functions and variables & global variables emulated on the PC host (as far as possible)...

	    Some new interface features, like wide editor window, enable\disable confirm individual files delete...	   	
		Improvement to SD init routine...

		First Release Version:
		
		As at 24 January 2011, by Mauro Grassi.
 9.91:  Fixed problem with bootloader activating if button is not pressed when power is applied. (NV)
        Added code so that micro will go to sleep in low battery condition even if no scripts are running. (NV)
 9.92:  Made device switch to internal oscillator immediately upon startup and wait for voltage to come up
        to an acceptable level (~1.6V). This prevents the logger drawing lots of power with a very low battery
        voltage (<1.2V). Also made it change to internal oscillator while sleeping for the same reason.
								       
/*** INCLUDES *******************************************************/

#include "Compiler.h"
#include "HardwareProfile.h"
#include "GenericTypeDefs.h"
#include <math.h>
#include "usb_device.h"
#include "usb.h"
#include "usb_function_generic.h"
#include "usb_config.h"
#include "clock.h"
#include "ff.h"
#include "diskio.h"
#include "mystring.h"
#include "main.h"
#include "capture.h"
#include "Delay.h"
#include "onewire.h"
#include "i2c.h"
#include "adc.h"
#include "spi.h"
#include "keys.h"
#include "serialUSB.h"
#include "led.h"
#include "uart.h"
#include "gps.h"
#include "vm.h"
#include "opcodes.h"
#include "io.h"
#include "capture.h"
#include "flash.h"
#include "boot.h"
#include "dee.h"
#include "bootwriter.h"

/***************************** CONFIGURATION **************************************************/

     #pragma config WDTEN 		= OFF          	//WDT disabled (enabled by SWDTEN bit)
     #pragma config PLLDIV 		= 5           	//Divide by 5 (20 MHz oscillator input)
     #pragma config XINST 		= ON        	//Extended instruction set?
     #pragma config CPUDIV 		= OSC1        	//No CPU system clock divide
     #pragma config CP0 		= OFF           //Program memory is not code-protected
     #pragma config OSC 		= HSPLL         //HS oscillator, PLL enabled, HSPLL used by USB
     #pragma config STVREN		= OFF			
     #pragma config FCMEN 		= OFF         	//Fail-Safe Clock Monitor disabled
     #pragma config IESO 		= OFF           //Two-Speed Start-up disabled
     #pragma config WDTPS 		= 32768        	//1:32768
     #pragma config DSWDTOSC 	= INTOSCREF 	//DSWDT uses INTOSC/INTRC as clock
     #pragma config RTCOSC 		= T1OSCREF    	//RTCC uses T1OSC/T1CKI as clock
     #pragma config DSBOREN 	= OFF        	//Zero-Power BOR disabled in Deep Sleep
     #pragma config DSWDTEN 	= OFF        	//Disabled
     #pragma config DSWDTPS 	= 8192       	//1:8,192 (8.5 seconds)
     #pragma config IOL1WAY 	= OFF        	//IOLOCK bit can be set and cleared
     #pragma config MSSP7B_EN 	= MSK7     		//7 Bit address masking
     #pragma config WPFP 		= PAGE_1    	//Write Protect Program Flash Page 0
     #pragma config WPEND 		= PAGE_0       	//Start protection at page 0
     #pragma config WPCFG 		= OFF          	//Write/Erase last page protect Disabled
     #pragma config WPDIS 		= OFF          	//WPFP[5:0], WPEND, and WPCFG bits ignored 
	 #pragma config ADCSEL		= BIT12			//Select 12 bit ADC operation!

void _boothook(void);
	 
#pragma udata maindata

unsigned char   		por;
unsigned char           firstPor;
unsigned char  			tempString[SIZE_OF_TEMPSTRING];
unsigned int 			transferCounter;
unsigned char*  		transferPtr;
unsigned long			devicePORs;
volatile unsigned char 	memoryCardSystemUp;
volatile unsigned char 	memoryCardEjected;
FIL	  					fsrc;
FILINFO					finfo;
DIR						fdir;
unsigned char	 	    fastTasks;
volatile unsigned short	vmTimer;
volatile unsigned char  captiveDevice;
volatile unsigned char	hostAlive;

/* These Are File Buffer Pointers */

short					fileIndex;
short					fileFull;
unsigned char			enterBootLoader;

/* These are large objects */

#pragma udata fsdata=0x260
//#pragma udata fsdata
FATFS 					fileSystem;

#pragma udata udldata=0x490
//#pragma udata udldata
DATA_LOGGER_CONTAINER	thisUDL;

#pragma udata 

/********************************** VARIABLES ******************************************************/
#pragma udata genmem

/** VECTOR REMAPPING ***********************************************/
#if defined(__18CXX)
	//On PIC18 devices, addresses 0x00, 0x08, and 0x18 are used for
	//the reset, high priority interrupt, and low priority interrupt
	//vectors.  However, the current Microchip USB bootloader 
	//examples are intended to occupy addresses 0x00-0x7FF or
	//0x00-0xFFF depending on which bootloader is used.  Therefore,
	//the bootloader code remaps these vectors to new locations
	//as indicated below.  This remapping is only necessary if you
	//wish to program the hex file generated from this project with
	//the USB bootloader.  If no bootloader is used, edit the
	//usb_config.h file and comment out the following defines:
	//#define PROGRAMMABLE_WITH_USB_HID_BOOTLOADER
	//#define PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x1000
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x1008
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x1018
	#elif defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)	
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x800
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x808
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x818
	#else	
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x00
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x08
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x18
	#endif
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	extern void _startup (void);        // See c018i.c in your C18 compiler dir
	#pragma code REMAPPED_RESET_VECTOR = REMAPPED_RESET_VECTOR_ADDRESS
	void _reset (void)
	{
	    _asm goto _startup _endasm
	}
	#endif
	#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS
	void Remapped_High_ISR (void)
	{
	     _asm goto YourHighPriorityISRCode _endasm
	}
	#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS
	void Remapped_Low_ISR (void)
	{
	     _asm goto YourLowPriorityISRCode _endasm
	}
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	//Note: If this project is built while one of the bootloaders has
	//been defined, but then the output hex file is not programmed with
	//the bootloader, addresses 0x08 and 0x18 would end up programmed with 0xFFFF.
	//As a result, if an actual interrupt was enabled and occured, the PC would jump
	//to 0x08 (or 0x18) and would begin executing "0xFFFF" (unprogrammed space).  This
	//executes as nop instructions, but the PC would eventually reach the REMAPPED_RESET_VECTOR_ADDRESS
	//(0x1000 or 0x800, depending upon bootloader), and would execute the "goto _startup".  This
	//would effective reset the application.
	
	//To fix this situation, we should always deliberately place a 
	//"goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS" at address 0x08, and a
	//"goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS" at address 0x18.  When the output
	//hex file of this project is programmed with the bootloader, these sections do not
	//get bootloaded (as they overlap the bootloader space).  If the output hex file is not
	//programmed using the bootloader, then the below goto instructions do get programmed,
	//and the hex file still works like normal.  The below section is only required to fix this
	//scenario.
	#pragma code HIGH_INTERRUPT_VECTOR = 0x08
	void High_ISR (void)
	{
	     _asm goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS _endasm
	}

	
	#pragma code LOW_INTERRUPT_VECTOR = 0x18
	void Low_ISR (void)
	{
	     _asm goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#endif	//end of "#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER)"

	#pragma code MAIN_APP_VECTOR = 0x10
	void mainApp(void)
	{
		_asm goto mainAppInternal _endasm
	}

	#pragma code usercode
	
	//These are your actual interrupt handling routines.
	#pragma interrupt YourHighPriorityISRCode
	void YourHighPriorityISRCode()
	{
		// Check which interrupt flag caused the interrupt.
		// Service the interrupt
		// Clear the interrupt flag
		// Etc.
		doCaptureInterrupt();	
  		
	    #if defined(USB_INTERRUPT)
	        USBDeviceTasks();
        #endif
	
		if((PIE5bits.TMR6IE)&&(PIR5bits.TMR6IF))
		{
			/* 
				must be called periodically from some real time tick source...
			*/	
			if(ledDivider>0)
			{
				ledDivider--;
			}
			else
			{
			ledDivider=(LED_DIVIDER-1);
		
			switch(ledState)
			{
				case LED_IDLE_STATE:
					break;
					
				case LED_ON_STATE:
					/* turn it ON */
					ledOnTime++;
					if(ledDelta>0)
					{
						ledTimer=ledDelta;
						if((LED1!=LED1_ON)||(LED1_TRIS!=0))
						{
							LED1=LED1_ON;
							LED1_TRIS=0;
						}	
						ledState=LED_ON_HOLD_OFF_STATE;
					} 
					else
					{
						ledState=LED_TURN_OFF_STATE;
					}	
					break;
									
				case LED_ON_HOLD_OFF_STATE:
					if((ledOnTime<LED_ON_HOLDOFF_MAX_TICKS)&&(ledTimer>0))
					{
						ledOnTime++;
						ledTimer--;
					}
					else
					{
						ledState=LED_TURN_OFF_STATE;
						ledOnTime++;	
					}
					break;
				
				default:
				case LED_TURN_OFF_STATE:
						if(T2CON!=0)
						{
						  LED1=!LED1_ON;
						  LED1_TRIS=0;
						}
						else
						{
						 LED1_TRIS=1;
						 LED1=!LED1_ON;
						}
						ledState=LED_OFF_HOLD_OFF_STATE;
						ledTimer=ledOnTime+LED_OFF_HOLDOFF_TICKS;
						break;
						
				case LED_OFF_HOLD_OFF_STATE:
					if(ledTimer>0)
					{
						ledTimer--;
					}
					else
					{
						ledTimer=ledOffTime;
						if(ledRepeat>0)ledRepeat--;
						ledState=LED_REDO_STATE;
					}
					break;
				
				case LED_REDO_STATE:
					if(ledTimer>0)
					{
						ledTimer--;	
					}
					else
					{
						ledOnTime=0;
						if(ledRepeat>0)
						{
							/* Redo! */
							ledTimer=0;
							ledState=LED_ON_STATE;
						}
						else
						{
							T6CON=0;
							PIR5bits.TMR6IF=0;
							PIE5bits.TMR6IE=0;
							ledState=LED_IDLE_STATE;
							ledDelta=0;
							ledRepeat=0;
						}	
					}
					break;
		 	}
		 	}
			PIR5bits.TMR6IF=0;  
		}
		else
		if(PIR5bits.TMR6IF)PIR5bits.TMR6IF=0;
 	 
 	 	 if((PIE2bits.CM1IE)&&(PIR2bits.CM1IF))
		{
			keyPressedCount=0;
			keyCountTimer=KEY_SCAN_PERIOD;
			if(keyInitialDebounce>0)
			{
	
			}
			else
			{
				keyState=KEYSTATE_INITIAL;
				setKeysComparator(0);				
				setKeysTimer(1);
			}
			PIR2bits.CM1IF=0;	
		}	
		else
		if(PIR2bits.CM1IF)PIR2bits.CM1IF=0;
		else
		if((PIE3bits.TMR4IE)&&(PIR3bits.TMR4IF))
		{
			if(keyCountTimer>0)keyCountTimer--;
			else
			{
				keyCountTimer=KEY_SCAN_PERIOD;
				switch(keyState)
				{
					case KEYSTATE_INITIAL:
					default:
						if(CMSTATbits.COUT1)
						{
							/* button pressed down */

							if(keyPressedCount<KEYPRESSED_LIMIT)
							{
								keyPressedCount++;
								keyCountTimer=KEY_SCAN_PERIOD;
							}
							else
							{
								keyState=KEYSTATE_ENTERKEY;
							}
						}
						else
						{
							/* button pressed up */
							keyState=KEYSTATE_ENTERKEY;
						}
						break;
									
					case KEYSTATE_ENTERKEY:
						if(keyPressedCount>=KEYPRESSED_LIMIT)
						{
							putKey(KEY_LONG);
							keyState=KEYSTATE_HOLDOFF;
							keyCountTimer=KEY_SCAN_PERIOD_HOLDOFF_LONG;
						}
						else
						if(keyPressedCount>=KEYPRESSED_THRESHOLD)
						{
							putKey(KEY_SHORT);
							keyState=KEYSTATE_HOLDOFF;
							keyCountTimer=KEY_SCAN_PERIOD_HOLDOFF_SHORT;
						}
						else
						{
							keyState=KEYSTATE_HOLDOFF;
							keyCountTimer=KEY_SCAN_PERIOD_HOLDOFF_SHORT;
						}
						break;
					
					case KEYSTATE_HOLDOFF:
						if(CMSTATbits.COUT1)
						{
							keyCountTimer=KEY_SCAN_PERIOD_HOLDOFF_SHORT;
						}
						else
						{
							setKeysTimer(0);
							setKeysComparator(1);
							keyState=KEYSTATE_INITIAL;
							keyCountTimer=KEY_SCAN_PERIOD_HOLDOFF_SHORT;
						}
						break;
				}
			}
			PIR3bits.TMR4IF=0;
	 	}
	 	else
	 	if(PIR3bits.TMR4IF)PIR3bits.TMR4IF=0;

		if((PIE1bits.TMR1IE)&&(PIR1bits.TMR1IF))
		{
			if(cardLED>0)
			{
				cardLED--;
			}
			
			if(fastTasks>0)fastTasks--;
			if(hostAlive>0)
			{
				hostAlive--;
			} 
			else 
			{
				captiveDevice=0;
				cardLED=0;
				fastTasks=0;
			}
			
			if(keyInitialDebounce>0)keyInitialDebounce--;
			timer1OverFlow++;
			
			
			/* Check if the alarm is in the past or the present (rather than the future!) */
				
			if(PIE3bits.RTCCIE)
			{
				/* alarm event occurred */
				timeAbsolute++;
				if(timeAbsolute>alarmAbsolute)
				{
					alarmEvent|=(ALARM_TRIGGER_EVENT | ALARM_FUTURE_EVENT);
					PIE3bits.RTCCIE=0;
					PIR3bits.RTCCIF=0;
					alarmAbsolute=1;
					timeAbsolute=0;
				}
			}	
			PIR1bits.TMR1IF=0;	
		}
		else
		if(PIR1bits.TMR1IF)PIR1bits.TMR1IF=0;

	}	//This return will be a "retfie fast", since this is in a #pragma interrupt section 

	#pragma interruptlow YourLowPriorityISRCode
	void YourLowPriorityISRCode()
	{
		/* Disable Nesting of Low Interrupts! */		
		#if(NO_INT_NESTING)
		INTCONbits.GIE=0;
		#elif(NO_LOW_INT_NESTING)
		INTCONbits.GIEL=0;
		#endif	 	
	 	
	 	if((PIE5bits.TMR8IE)&&(PIR5bits.TMR8IF))
		{
			if(vmTimer>0)
			{
				if(vmTimer<0xFFFF)vmTimer++;
			}
			PIR5bits.TMR8IF=0;
		}
		else
		if(PIR5bits.TMR8IF)PIR5bits.TMR8IF=0;
		
		if((PIE5bits.CM3IE)&&(PIR5bits.CM3IF))
		{
			memoryCardEjected=1;
			PIE5bits.CM3IE=0;
			PIR5bits.CM3IF=0;
		}
		
		#if 0
		if((PIE3bits.RC2IE)&&(PIR3bits.RC2IF))
		{
			receiveCharVM(&thisUDL.dataLogger.ven.cVM);
			PIR3bits.RC2IF=0;
		}
		else
		if(PIR3bits.RC2IF)PIR3bits.RC2IF=0;
        #endif
		//Check which interrupt flag caused the interrupt.
		//Service the interrupt
		//Clear the interrupt flag
		//Etc.
		
		if((PIE3bits.RTCCIE)&&(PIR3bits.RTCCIF))
		{
			/* alarm event occurred */
			alarmEvent|=ALARM_TRIGGER_EVENT;
			PIE3bits.RTCCIE=0;
			PIR3bits.RTCCIF=0;
			alarmAbsolute=1;
			timeAbsolute=0;
		}
		else
		if(PIR3bits.RTCCIF)PIR3bits.RTCCIF=0;
		
		#if(NO_INT_NESTING)
		INTCONbits.GIE=1;
		#elif(NO_LOW_INT_NESTING)
		INTCONbits.GIEL=1;
		#endif
		
	}	//This return will be a "retfie", since this is in a #pragma interruptlow section 

#elif defined(__C30__)
    #if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
    #endif
#endif

/** DECLARATIONS ***************************************************/
#pragma code usercode

#if 0
char* putFileName(char* inptr, const rom char* srcptr)
{
	int i;
	i=0;
	while((i<(VM_FILENAME_MAX-1))&&((*srcptr)!='\0'))
	{
		*inptr++=*srcptr++;
		i++;
	}
	*inptr='\0';
	return inptr;
}
#endif

void logStringRom(const rom unsigned char* instring, unsigned char function)
{
	outputFunctionVM(0, function);
	outputSROMVM(0, instring);
}

void logString(unsigned char* instring, unsigned char function)
{
	outputFunctionVM(0, function);
	outputSVM(0, instring);
}

void logStringOnly(unsigned char* instring)
{
	outputSVM(0, instring);
}

void logF(const rom unsigned char* instring, unsigned char function, float f, unsigned char numdec)
{
	logStringRom(instring, function);
	uftoa(tempString, f, numdec);
	outputSVM(0, tempString);
}

VM_RETURN_TYPE flushLog(void)
{
	VM_RETURN_TYPE result;
	result=writeCacheToDisk(&thisUDL.dataLogger.ven.vmLogFileCache, thisUDL.dataLogger.ven.vmLogFileCache.cacheStart, thisUDL.dataLogger.ven.vmLogFileCache.bottomAddress, thisUDL.dataLogger.ven.vmLogFileCache.topAddress);
	return result;
}

VM_RETURN_TYPE deleteLog(void)
{
	f_unlink((char*)&thisUDL.dataLogger.ven.vmLogFileName[0]);
	return initCache(&thisUDL.dataLogger.ven.vmLogFileCache, thisUDL.dataLogger.ven.vmLogFileCache.cacheMode, FILE_CACHE_ID + 0, VM_SYSTEM_LOGFILE_CACHE_SIZE, MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_SYSTEM_OFFSET], &thisUDL.dataLogger.ven.vmLogFileName[0]);
}

VM_RETURN_TYPE restartLog(void)
{
	return initCache(&thisUDL.dataLogger.ven.vmLogFileCache, thisUDL.dataLogger.ven.vmLogFileCache.cacheMode, FILE_CACHE_ID + 0, VM_SYSTEM_LOGFILE_CACHE_SIZE, MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_SYSTEM_OFFSET], &thisUDL.dataLogger.ven.vmLogFileName[0]);
}

void setDevicePORs(unsigned long x)
{
	unsigned char* ptr;
	devicePORs=x;
	ptr=(unsigned char*)&devicePORs;
	DataEEWrite(*ptr++, EE_PORS_ADDR);
	DataEEWrite(*ptr++, EE_PORS_ADDR+1);
	DataEEWrite(*ptr++, EE_PORS_ADDR+2);
	DataEEWrite(*ptr++, EE_PORS_ADDR+3);
}

unsigned long getDevicePORs(void)
{	
	unsigned long  x;
	unsigned char* ptr;
	x=0;
	ptr=(unsigned char*)&x;
	*ptr++=DataEERead(EE_PORS_ADDR);
	*ptr++=DataEERead(EE_PORS_ADDR+1);
	*ptr++=DataEERead(EE_PORS_ADDR+2);
	*ptr++=DataEERead(EE_PORS_ADDR+3);
	return x;
}

void flushSettings(void)
{
	saveSettingsFileTask();
	saveSettingsTask();
}

void initSystemUART(DATA_LOGGER* dl)
{	
	#if 0
		// the following is useful during development...
		dl->ven.systemState.serialPort.baudRate=DEFAULT_BAUD_RATE/BAUD_RATE_FACTOR;
		dl->ven.systemState.serialPort.txrxpin=0x54;
		dl->ven.systemState.serialPort.mode=0;
		initUARTState(0, &dl->ven.systemState.serialPort);
	#else
		dl->ven.systemState.serialPort.baudRate=DEFAULT_BAUD_RATE/BAUD_RATE_FACTOR;
		dl->ven.systemState.serialPort.txrxpin=0x54;
		dl->ven.systemState.serialPort.mode=NO_UART_MODE;
		initUARTState(0, &dl->ven.systemState.serialPort);
	#endif
}

void setPORSettings(DATA_LOGGER* dl)
{
	dl->lastSyncTime.show=0;
	dl->lastSyncTime.hours=0;
	dl->lastSyncTime.secs=0;
	dl->lastSyncTime.mins=0;
	dl->lastSyncTime.hours=0;
	dl->lastSyncTime.day=1;
	dl->lastSyncTime.month=1;
	dl->lastSyncTime.year=2011;
	dl->lastSyncTime.wday=5;
	dl->lastSyncTime.updated=0;
}

void setDefaultSettings(DATA_LOGGER* dl)
{
	dl->pors=PORS_START;
	dl->modes=DEFAULT_DATA_LOGGER_MODE;
	dl->batteryProtectionVoltage=DEFAULT_PROTECTION_VOLTAGE;
	dl->ven.vmLogFileName[0]='\0';
	initVMEnvironment(&dl->ven);
	initSystemUART(dl);
	igetTime((TIME_T*)&systemTime);
	copyTime(&dl->lastSyncTime, (TIME_T*)&systemTime);
	setPORSettings(dl);
	dl->dirty|=DL_DIRTY;
}

unsigned char* getFileNameDataLogger(DATA_LOGGER* dl, unsigned char* fileN)
{
	fileN[0]='d';
	fileN[1]='l';
	fileN[2]='.';
	fileN[3]='b';
	fileN[4]='i';
	fileN[5]='n';
	fileN[6]='\0';
	return &fileN[0];
}

void restoreSettings(DATA_LOGGER* dl)
{
	unsigned char  	CRC;

	unsigned int 	i;
	unsigned char* 	ptr;
	unsigned int   	numWriteSettings;
	
	numWriteSettings=NUM_WRITE_SETTINGS;
	CRC=0;
	/* Start from index 1 since 0 is the dirty byte */
	i=1;
	ptr=(unsigned char*)dl;
	ptr+=i;
	while(i<numWriteSettings)
	{
		*ptr=DataEERead(i+EE_DATA_LOGGER_ADDR);
		CRC^=(unsigned char)(*ptr);
		ptr++;

		#if 0
		putrsUART("Restore CRC : ");
		disWUART(CRC);
		putrsUART("\r\n");
		#endif
		i++;
	}
	CRC^=DataEERead(numWriteSettings+EE_DATA_LOGGER_ADDR);
	if(CRC!=0)
	{
		/* There Was a CRC Error ! */
		setDefaultSettings(dl);
		CRC=0;
	}
	dl->CRC=CRC;
}

unsigned char saveSettingsFile(DATA_LOGGER* dl)
{
	/* Save Data Logger Settings To File, if possible */
	unsigned int   sz;
	unsigned int   uz;
	unsigned char  CRC;
	FRESULT		   fresult;
	unsigned char* ptr;
	unsigned char  result;
	FIL	  		   ftemp;
	
	result=0;
	if(dl->dirty & DL_FILE_DIRTY)
	{
		f_close(&ftemp);	
		getFileNameDataLogger(dl, fileName);
		if((memoryCardSystemUpMacro())&&(f_open(&ftemp, (char*)fileName, FA_WRITE | FA_CREATE_ALWAYS)==FR_OK))
		{
			sz=sizeof(DATA_LOGGER);
			ptr=(unsigned char*)dl;
			CRC=0;
			for(uz=0; uz<(sz-1); uz++)
			{
				CRC^=ptr[uz];
			}
			dl->CRC=CRC;
			fresult=f_write(&ftemp, (void*)dl, sz, &uz);
			if(fresult==FR_OK)
 	 		{	
				if(sz==uz)result=1;
	 		}
	 	}
		f_close(&ftemp);
		dl->dirty&=~DL_FILE_DIRTY;
	}
	return result;
}

unsigned char restoreSettingsFile(DATA_LOGGER* dl)
{
	unsigned char  result;
	/* Restore Data Logger Settings From File, if possible */
	unsigned char  CRC;
	FRESULT		   fresult;
	unsigned char* ptr;
	unsigned int   sz;
	unsigned int   uz;
	FIL	  		   ftemp;
	
	result=0;
	f_close(&ftemp);
	getFileNameDataLogger(dl, fileName);
	if((memoryCardSystemUpMacro())&&(f_open(&ftemp, (char*)fileName, FA_READ | FA_OPEN_EXISTING)==FR_OK))
	{
			sz=sizeof(DATA_LOGGER);
			fresult=f_read(&ftemp, (void*)dl, sz, &uz);
			if(fresult==FR_OK)
 	 		{	
				if(sz==uz)
				{				
					ptr=(unsigned char*)dl;
					CRC=0;
					for(uz=0; uz<sz; uz++)
					{
						CRC^=ptr[uz];
					}
					if(CRC==0)result=1;		
				}
	 		}
	}
	f_close(&ftemp);
	return result;
}

unsigned int saveSettings(DATA_LOGGER* dl)
{
	/* Returns the number of settings saved... */
	
	unsigned char* ptr;
	unsigned char  c;
	unsigned char  CRC;
	unsigned int   numWriteSettings;
	unsigned int   written;
	unsigned int   i;
	unsigned char  k;
	
	written=0;	
	if(dl->dirty & DL_EE_DIRTY)
	{	
		numWriteSettings=NUM_WRITE_SETTINGS;
		CRC=0;
		/* Start The Index from i=1 */
		i=1;
		ptr=(unsigned char*)dl;
		ptr+=i;
		while(i<(numWriteSettings))
		{
			c=DataEERead(i+EE_DATA_LOGGER_ADDR);
			
			if(c!=(*ptr))
			{
			
			#if 0
			putrsUART("At: ");
			disAUART((unsigned char)i+EE_DATA_LOGGER_ADDR);
			putrsUART(" To Save: ");
			disAUART((unsigned char)*ptr);
			putrsUART(" EE: ");
			disAUART((unsigned char)c);
			putrsUART("\r\n");
			#endif
			
			CRC^=(unsigned char)(*ptr);
			DataEEWrite(*ptr, i+EE_DATA_LOGGER_ADDR);
			written++;
			}
			else
			{
			CRC^=(unsigned char)c;
			}
			#if 0
				putrsUART("Save CRC : ");
				disWUART(CRC);
				putrsUART("\r\n");
			#endif
			ptr++;
			i++;
		}
		if(written>0)DataEEWrite(CRC, numWriteSettings+EE_DATA_LOGGER_ADDR);
		dl->CRC=0;
		dl->dirty&=~DL_EE_DIRTY;
	}
	return written;
}

unsigned int syncSettings(DATA_LOGGER* dl)
{
	/* 
		Make the EE settings and the File settings the same... 
		
		If the EE settings disagree with the file settings, we take the
		file settings as the correct settings, because these can differ
		depending on the memory card, and so are more local...
		
		Returns the number of setings written if there was a disagreement...
				
	*/
	
	unsigned int 	i;
	unsigned char* 	ptr;
	unsigned char  	CRC;
	unsigned int   	numWriteSettings;
	unsigned char   result;
	unsigned int	written;
	
	written=0;
	if(memoryCardSystemUpMacro())result=restoreSettingsFile(dl); else result=0;
	if(result)
	{
		/*  
		
			The settings were restored properly from the file, including
			CRC checking, so continue to save to EE... 
			
		*/
		dl->dirty&=~DL_FILE_DIRTY;
		dl->dirty|=DL_EE_DIRTY;
		written=saveSettings(dl);	
		
		#if(USE_EEPROM)
			/* If using FLASH based EEPROM emulation, these settings take precedence... */
			restoreSettings(dl);
		#else
			/* If using memory card based EEPROM emulation, check the settings against EE settings */
		#endif
	}
	else
	{
		/* 
			The settings were not restored from the file properly, therefore
			get them from EE...
			
		*/
		if(memoryCardSystemUpMacro())
		{
			/* 
				restore from EE then... Note that we restore the defaults first because EE does not
				store ALL the settings...
				
			*/
			setDefaultSettings(dl);
			restoreSettings(dl);	
		}
		else
		{
			/* no memory card, do nothing then! */
		}
	}
	return written;
}

void saveSettingsTask(void)
{
	unsigned int written;
	
	if(thisUDL.dataLogger.dirty & DL_EE_DIRTY)
	{
		written=saveSettings(&thisUDL.dataLogger);
		if((written>0)&&(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE))
		{	   	
		   	logF((const rom unsigned char*)": Saved Settings: ", DEFAULT_LOG_PF, (float)written, 0);
		   	logStringRom((const rom unsigned char*)".\r\n", PF_NOP);
		   	flushLog();
   		}   	
	}
}

void saveSettingsFileTask(void)
{
	unsigned char result;
	
	if(thisUDL.dataLogger.dirty & DL_FILE_DIRTY)
	{
		result=saveSettingsFile(&thisUDL.dataLogger);
		if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
		{	   	
		   	logStringRom((const rom unsigned char*)": Saved Settings To File: ", DEFAULT_LOG_PF);
		   	getResultString(tempString, result);
		   	logStringOnly(tempString);
		   	logStringRom((const rom unsigned char*)".\r\n", PF_NOP);
		   	
		   	if(thisUDL.dataLogger.ven.vmMode & VME_REWIND)
		   	{
			   	logStringRom((const rom unsigned char*)": Reset VM Environment.\r\n", DEFAULT_LOG_PF);
			   	thisUDL.dataLogger.ven.vmMode&=~VME_REWIND;
		    }	
		   	flushLog();
   		}		
	}
}

void syncSettingsTask(void)
{
	unsigned int written;	

	written=syncSettings(&thisUDL.dataLogger);
	if((written>0)&&(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE))
	{	   	
	   	logF((const rom unsigned char*)": Sync Settings: ", DEFAULT_LOG_PF, (float)written, 0);
	   	logStringRom((const rom unsigned char*)".\r\n", PF_NOP);
	   	flushLog();
   	}   
}   	

void logVMStats(void)
{
	unsigned char index;
	
	if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
	{
		   	if(thisUDL.dataLogger.ven.vmNum>0)
		   	{
		   	logF((const rom unsigned char*)": VM(s) Running: ", DEFAULT_LOG_PF, (float)getNumVMRunning(&thisUDL.dataLogger.ven), 0);
		   	logF((const rom unsigned char*)" of ", PF_NOP, (float)thisUDL.dataLogger.ven.vmNum, 0);
		   	logStringRom((const rom unsigned char*)".\r\n", PF_NOP);
			logStringRom((const rom unsigned char*)": The Following VM(s) Are Loaded: { ", DEFAULT_LOG_PF);
		   	index=0;
		   	while(index<(unsigned char)thisUDL.dataLogger.ven.vmNum)
		   	{
		   		getScriptName(tempString, thisUDL.dataLogger.ven.vmID[index], SIZE_OF_TEMPSTRING);
		   		logStringOnly(tempString);
		   		if(index!=(thisUDL.dataLogger.ven.vmNum-1))logStringRom((const rom unsigned char*)", ", PF_NOP);
		   		index++;
		   	}
		   	logStringRom((const rom unsigned char*)" }\r\n", PF_NOP);	   	
		   	}
		   	else
		   	{
		   		logStringRom((const rom unsigned char*)": No VM(s) Loaded!\r\n", DEFAULT_LOG_PF);
		   	}
	}	
}   

void logHeader(void)
{
	unsigned char temp;
		
	temp=thisUDL.dataLogger.modes;
   	thisUDL.dataLogger.modes|=DEFAULT_DATA_LOGGER_LOGGING_MODE;	
   	if(memoryCardSystemUpMacro())
   	{
   	   		logF((const rom unsigned char*)": USB Data Logger Version: ", DEFAULT_LOG_PF, (float)FIRMWARE_VERSION, 2);
		   	logF((const rom unsigned char*)". Global PORs: ", PF_NOP, (float)devicePORs, 0);
		   	logF((const rom unsigned char*)". Local PORs: ", PF_NOP, (float)thisUDL.dataLogger.pors, 0);
		   	logStringRom((const rom unsigned char*)".\r\n", PF_NOP);
		
			if(temp & SYSTEM_LOGGING_MODE)
			{   			   	
		 	(void)updateFreeSize(&cardInfo);	   	
			logStringRom((const rom unsigned char*)": Memory Card Detected, Total Size: ", DEFAULT_LOG_PF);
			getMemoryCardTotalSize(&cardInfo,(unsigned char*)tempString);
			logStringOnly(tempString);
		 	logStringRom((const rom unsigned char*)" Free Size: ", PF_NOP);
		  	getMemoryCardFreeSize(&cardInfo,(unsigned char*)tempString);
		  	logStringOnly(tempString);
		   	logStringRom((const rom unsigned char*)".\r\n", PF_NOP);
			logVMStats();
		 	}  	
		 	else
		 	{
		 		logStringRom((const rom unsigned char*)": System Log Disabled.\r\n", DEFAULT_LOG_PF);
		 	}
		   	flushLog();	
  	}
   	else
   	{
  
   	}
	thisUDL.dataLogger.modes=temp;
}

unsigned char isPOR(void)
{
	if(RCONbits.POR==0)
	{
		RCONbits.POR=1;
		return 1;
	}
	else
	{
		return 0;
	}
}

void initSettings(DATA_LOGGER* dl)
{
	unsigned char x;
	unsigned int i;
	unsigned char* ptr;
	
    dataEEFlags.val=0;
    DataEEInit();
    dataEEFlags.val=0; 
   	devicePORs=getDevicePORs();
	if(devicePORs==0xFFFFFFFF)setDevicePORs(PORS_START);
	else
	{
		devicePORs++;
   		setDevicePORs(devicePORs);
   	}
    x=DataEERead(EE_MAGIC_ADDR);
    if(x==EE_MAGIC_VALUE)
    {
		syncSettings(dl);
		clearVMEnvironment(&dl->ven);	
    	if(por)setPORSettings(dl);
    	dl->dirty|=DL_DIRTY;
    	saveSettings(dl);
    	saveSettingsFile(dl);
    }
    else
    {
		setDefaultSettings(dl);
		dl->dirty|=DL_DIRTY;
		saveSettings(dl);
		saveSettingsFile(dl);
		DataEEWrite(EE_MAGIC_VALUE, EE_MAGIC_ADDR);
    }
}

#if 0
void showBuffer(unsigned char* data, unsigned int count)
{
	int i;
	i=0;
	while(count--)
	{
		disAUART(*data++);
		putcUART('.');
		i++;
		if((i % 16)==0)putrsUART("\r\n");
	}
	putrsUART("\r\n");
}
#endif

void goToSleep(unsigned char c)
{
	/* If c is non zero, go to sleep, otherwise, wake from sleep */
	
	if(c)
	{
		/* go to sleep */
		INTCONbits.GIE=0;
		INTCONbits.GIEL=0;
		/* Turn Off All Non Essential Interrupts! */		
		PIE1bits.TMR1IE=0;
		PIE2bits.TMR3IE=0;
		PIE5bits.TMR6IE=0;
		PIE5bits.TMR5IE=0;
		PIE5bits.TMR8IE=0;	
	 	PIE3bits.RC2IE=0;
	 	PIE1bits.CCP1IE=0;
		PIE2bits.CCP2IE=0;
		PIE4bits.CCP3IE=0;
		PIE4bits.CCP4IE=0;
		PIE4bits.CCP5IE=0;
		PIE4bits.CCP6IE=0;
		PIE4bits.CCP7IE=0;
	 	
	 	/* Send All Systems To Sleep, one by one */
		/* USB System */
		USBSoftDetach();
		/* SPI System */
		fileIndex=0;
		fileFull=0;

		memoryCardSystemUp=0;
		closeSPI();
		cardInfo.cERROR=ERROR_CLOSED_CARD;
		cardInfo.SIZEFREE=0;
		/* The Keys System */
		//setKeysTimer(0);
		//setKeysComparator(0);
		//powerKeys(0);
		/* The LED System */
		powerLED(0);
		/* The ADC System */
		adcSave=adcOpen;
		reopenCloseADCObject(adcOpen, 0);
		powerADC(0);
		/* The Capture System */
		reopenCloseCaptureObject(0);
		powerCapture(0);
		/* The UART */
		
		clearUARTDescriptor(&thisUDL.dataLogger.ven.systemState.serialPort);
		initUARTState(0, &thisUDL.dataLogger.ven.systemState.serialPort);
		/* Any Remaining general IO pins go to high impedance */
		/* Set All Interrupt Sources that can wake us up from sleep */
		//initKeys();
			
		CM1CON=0x8D;
		PIR2bits.CM1IF=0;
		PIE2bits.CM1IE=1;
		/* This is for the memory card */
		/* Check Memory Card Status */
		
		if(SDSENSE==SDSENSE_ON)
		{
			powerSwitchComparator(1, 0);
		}
		else 
		{
			powerSwitchComparator(1, 1);
		}
		INTCONbits.GIEL=1;
		INTCONbits.GIE=1;
	
		/* Actually go to sleep... */
//		OSCCONbits.IDLEN=0;
		OSCCON = 0x0B;
		DSCONHbits.DSEN=0;
		Sleep();
		OSCCON = 0x68;
		INTCONbits.GIE=0;
		INTCONbits.GIEL=0;
	}
	else
	{
		/* Wake up all systems from sleep, one by one */
		/* SPI System, self recovers */
		/* The Keys System */
		// initKeys();
		/* The LED System */
		initLED();
		/* The ADC System */
		adcOpen=0;
		powerADC(1);
		initADC();	
		reopenCloseADCObject(adcSave, 1);
		/* The Capture System */
		initCapture();
		reopenCloseCaptureObject(1);
		/* The UART */
		initSystemUART(&thisUDL.dataLogger);
		/* Re arm interrupts */	
		PIE1bits.TMR1IE=1;
		powerSwitchComparator(0, 0);
		USBDeviceAttach();
		INTCONbits.GIE=1;
		INTCONbits.GIEL=1;
	}
}

void sleepTime(unsigned char switchOffPorts)
{
	/* 
		If override==1 then it is possible to override the sleep command by holding
		down switch S2 - this can be useful with very slow PCs for installing the driver ! 
	
	*/
	unsigned char x;
	
	if(hostAlive==0)
	{	
		if(switchOffPorts)
		{
			closeAllIO();
		}	
		f_close(&fsrc);
   		if(memoryCardSystemUpMacro())
		{
			flushLog();
			saveSettingsFileTask();
			#if(USE_EEPROM)
			
			#else
			saveSettingsTask();
			#endif
		}
		
		#if(USE_EEPROM)
			saveSettingsTask();
		#endif
		
		goToSleep(1);
		goToSleep(0);
		memoryCardTask();
		hostAlive=HOST_ALIVE_ADDITIONAL_TIMEOUT;
		if(switchOffPorts)
		{
			openAllIO(ioOpen);
		}
	}
}

#if 0
unsigned char* showDataLogger(DATA_LOGGER* dl)
{
	unsigned char* outstr;
	
	putrsUART((const rom unsigned char*)"Dirty: ");
	outstr=uftoa(tempString, (float)dl->dirty, 0);
	putsUART(tempString);
	
	putrsUART((const rom unsigned char*)" Modes: ");
	outstr=uftoa(tempString, (float)dl->modes, 0);
	putsUART(tempString);
	
	putrsUART((const rom unsigned char*)" PORs: ");
	outstr=uftoa(tempString, (float)dl->pors, 0);
	putsUART(tempString);

	putrsUART((const rom unsigned char*)" BattV: ");
	outstr=uftoa(tempString, (float)dl->batteryProtectionVoltage, 2);
	putsUART(tempString);
	
	return outstr;
}
#endif

void doCheckKeys(unsigned char checkKeys)
{
	unsigned char key;
	
	if(checkKeys)
	{
		key=getKey();
        if(key!=NO_KEY)hostAlive=HOST_ALIVE_ADDITIONAL_TIMEOUT;
        if((checkKeys & 1)&&(key==KEY_SHORT))
		{		
			/* Show that it is running */
			if(getNumVMRunning(&thisUDL.dataLogger.ven)<=0)
			{
			  	internalSetLED(3, STANDARD_LED_ON_MS, STANDARD_LED_OFF_MS);  			  
			}
			else 
			{
				setLED(STANDARD_LED_ON_MS);
			}
						
			#if 0
				putrsUART("STATE: ");
				disWUART(thisUDL.dataLogger.ven.vmState);
				putrsUART("\r\n");
			#endif
		}
		else
		if((checkKeys & 2)&&(key==KEY_LONG))
		{
			thisUDL.dataLogger.ven.vmMode|=VME_HOLD;
		}	
	}
}

void restartVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	clearVMEnvironment(ven);
	destroyAllVM(ven, &firstPor);
	ven->vmMode&=~(VME_RESTART | VME_HOLD);
	ven->vmState=VME_START_CYCLE;
 	flushSettings();
}

#pragma code usercode

void mainAppInternal(void)
{   
	#if(MAIN_DEBUG)
		unsigned char   ostate;
	#endif
	
	unsigned short	execLimit;
	unsigned char 	cont;
	unsigned char   checkKeys;
	VM_RETURN_TYPE  result;
	unsigned int    oldSecsCaptive;
	unsigned int    oldSecsGeneral;
	unsigned char   key;
	unsigned char   slowPeriod;
	float 			f;
	unsigned char   numRunning;
	
	hostAlive=0;
    captiveDevice=1;
  	cardLED=0;  
	memoryCardSystemUp=0;
	memoryCardEjected=0;
	ioOpen=0xFF;
	
	keyInitialDebounce=KEY_INITIAL_DEBOUNCE;
	initADC();
 	initKeys();

    clearCapture();
 	initCapture();
   	initTransfers();	
	clearUARTDescriptor(&thisUDL.dataLogger.ven.systemState.serialPort);
	initUARTState(0, &thisUDL.dataLogger.ven.systemState.serialPort);

	/* This can only be called once! */
	por=isPOR();
	
	if(por)
	{
		timeUp=0;
		firstPor=1;
	}
	
	initRTCC((TIME_T*)&systemTime);
	initLED();
    initSerialUSB();
    
   	setDefaultSettings(&thisUDL.dataLogger);
    initMemoryCardSystem(&cardInfo);
		
	initSettings(&thisUDL.dataLogger);
	initSystemUART(&thisUDL.dataLogger);
 
	/* Enable Interrupts */
	
	USBDeviceInit();
	#if defined(USB_INTERRUPT)
        USBDeviceAttach();
    #endif
    INTCONbits.GIE=1;
    INTCONbits.GIEL=1;
    
	/* Indicate the initial Startup! */

   	oldSecsCaptive=0;
   	oldSecsGeneral=0;
	key=NO_KEY;
	slowPeriod=0;
   	fastTasks=0;
    
    if(SDSENSE==SDSENSE_ON)
    {
	    /* If a memory Card is in the socket, go for a shorter startup time! */
    	hostAlive=HOST_ALIVE_INITIAL_TIMEOUT_SHORT;    
    }
    else
    {
	    /* Otherwise go for a long startup, to allow slow PCs to install the driver properly... */
    	hostAlive=HOST_ALIVE_INITIAL_TIMEOUT_LONG;
    }

  	setLED(STANDARD_LED_ON_MS);

	logHeader();
    (void)destroyAllVM(&thisUDL.dataLogger.ven, &firstPor);
	(void)rewindVMEnvironment(&thisUDL.dataLogger.ven);
	/* Delete the temp memory File */
	enterBootLoader=0;

   	while(1)
    {       
	    do 
	    { 
	
		memoryCardTask();
		memoryCardEmptyPowerSaverTask();
		
		if(fastTasks==0)
		{
		saveSettingsFileTask();
		if(oldSecsCaptive!=timer1OverFlow)
        {
	       	if(memoryCardSystemUp)
			{
			
			}
			else
			{
				if(SDSENSE==SDSENSE_ON)
				{
				
				}
				else
				{
					if(memoryCardEjected==0)
					{
						memoryCardEjected=1;
						hostAlive=HOST_ALIVE_ADDITIONAL_TIMEOUT;
						cardInfo.cERROR=ERROR_CLOSED_CARD;
					}
				}
			}
			
			if((hostAlive>0)&&(captiveDevice))
	        {	        
	       	  /* These tasks are done periodically if connected to the PC host */
	       	  key=getKey();
	       	  if(SDSENSE==SDSENSE_ON)
	       	  {
	       	  
	       	  }
	       	  else
	       	  {
	       		  if(key==KEY_SHORT)
	     	  	  {
	       	  		  hostAlive=HOST_ALIVE_ADDITIONAL_TIMEOUT;	       	  		
	      	 	  }
	       		  else
	       		  if(key==KEY_LONG)
	       		  {
		       		  /* you can add to the timeout here, if no memory card is present! */
		    	   	  hostAlive+=HOST_ALIVE_ADDITIONAL_TIMEOUT;
	       	  	  }		
		        }
		      	flushLog();	       	  		
  			  	setLED(STANDARD_LED_ON_MS);
	     	}
	     	else
	     	{
	       	  if(hostAlive>HOST_ALIVE_ADDITIONAL_TIMEOUT)
	       	  {
		    	numRunning=(unsigned char)getNumVMRunning(&thisUDL.dataLogger.ven);
				if(numRunning>0)internalSetLED(1, STANDARD_LED_ON_MS, STANDARD_LED_OFF_MS);
				else internalSetLED(3, STANDARD_LED_ON_MS, STANDARD_LED_OFF_MS);
  			  }
  		 	}
    		oldSecsCaptive=timer1OverFlow;    
	    }
	    	    	
	 	}
	 	else
	 	{
		 	if(cardLED>1)
			{
				internalSetLED(1, STANDARD_LED_ON_SHORT_MS, STANDARD_LED_OFF_SHORT_MS);
  			}
  			else
  			{ 			
  			
  			}
	 	}
	
		 if((USBDeviceState<CONFIGURED_STATE)||(USBSuspendControl==1))
	     {
	       	 
	     }
	     else
	     {
		       	 while(!USBHandleBusy(USBGenericOutHandle))
		       	 {
		       	 	ProcessIOTask(1);
					#if(BOOT_WRITER)
				 
				 	if(enterBootLoader)
				 	{
					 	 flushLog();
					 	 ejectCard();
					 	 while(USBHandleBusy(USBGenericInHandle));
					 	 USBSoftDetach();
						
						 if(bootFunction)
						 {
						 	/* Enter the USB BootLoader */
						 	_asm goto _boothook _endasm
						 }
						 else
						 {
							/* Enter the Serial BootLoader */					
							doBootLoaderUpdate();
						 }
				 	}
				 
				 	#endif
		         }        
	     }
	    }
	    while((hostAlive>0)&&(captiveDevice));

        while((thisUDL.dataLogger.modes & BATTERY_PROTECTION_MODE)&&((f=getADCP())<thisUDL.dataLogger.batteryProtectionVoltage))
			sleepTime(1);

	    if((memoryCardSystemUpMacro())&&(!((hostAlive>0)&&(captiveDevice))))
	    {
	 		/* VM Environment States */
	    	thisUDL.dataLogger.ven.vmMode&=~VME_VEDEBUG;
	   		execLimit=VME_SUPER_TASK_LIMIT;	
			cont=1;
			result=VM_OK;
			key=NO_KEY;
			while((cont)&&(result==VM_OK)&&(execLimit>0)&&(memoryCardSystemUpMacro()))
			{
				cont=1;
				if(oldSecsGeneral!=timer1OverFlow)
			    {
				    	memoryCardTask();
				    	
						if(slowPeriod>0)
						{
							slowPeriod--;
						}
						else
						{
				      	   	/* Check Battery Level */
				      	   	if(hostAlive==0)
				      	   	{
					        if((thisUDL.dataLogger.modes & BATTERY_PROTECTION_MODE)&&((f=getADCP())<thisUDL.dataLogger.batteryProtectionVoltage))
					        {
					           		if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
				   					{
				   						logF((const rom unsigned char*)": Battery Too Low. Voltage: ", DEFAULT_LOG_PF, f, 2);
				   						logF((const rom unsigned char*)" V, Threshold: ", PF_NOP, thisUDL.dataLogger.batteryProtectionVoltage, 2);
				   						logStringRom((const rom unsigned char*)" V. Sleeping...\r\n", PF_NOP);
				   						flushLog();
				   					}
				   					sleepTime(1);
				   					slowPeriod=SLOW_PERIOD_BATTERY_LOW_RECOVERY_SECS;
				   					restartVMEnvironment(&thisUDL.dataLogger.ven);
				   		     }
					         else
					         {
					     	   	saveSettingsTask();
				       	  		saveSettingsFileTask();
				  		  		slowPeriod=SLOW_PERIOD_TASKS_SECS;
							 }
							 }
				  		}     	 
				        oldSecsGeneral=timer1OverFlow;    
				 }
				    
			     if((USBDeviceState==CONFIGURED_STATE)&&(USBSuspendControl!=1)&&(!USBHandleBusy(USBGenericOutHandle)))
				 {
						ProcessIOTask(1);
						cont=0;
			 	 }
			 	 else	
			 	 if(SDSENSE==SDSENSE_ON)
			 	 {
			 		execLimit--;
					result=VMEnvironmentTask(&thisUDL.dataLogger.ven);
					checkKeys=0;
					
					#if (MAIN_DEBUG)
					if(ostate!=thisUDL.dataLogger.ven.vmState)
					{
						putrsUART("STATE: ");
			 			disWUART(thisUDL.dataLogger.ven.vmState);
						putrsUART("\r\n");
						ostate=thisUDL.dataLogger.ven.vmState;
					}
					#endif
					
					switch(thisUDL.dataLogger.ven.vmState)
					{
						default:		
						case VME_START_CYCLE:
						case VME_GET_NEXT_VM:
						case VME_SUSPEND_VM:
						case VME_STEP_VM:
							if(thisUDL.dataLogger.ven.vmNum<=1)
							{
								checkKeys=1;
							}
							break;
												
						case VME_END_CYCLE:
						case VME_WAITING_STATE:
						case VME_TIMEOUT_CYCLE: 
						case VME_TIMEOUT_WAIT_STATE:
							capture_FullTask(); 
							checkKeys=1;
							break;
							
						case VME_HOLDING_STATE:
							if(thisUDL.dataLogger.ven.vmNum>0)
							{
							if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
							{	   	
								logStringRom((const rom unsigned char*)": Holding.\r\n", DEFAULT_LOG_PF);
							   	flushLog();
					   		}  
   							}		  	
   						  	internalSetLED(3, STANDARD_LED_ON_MS, STANDARD_LED_OFF_MS);
					   		thisUDL.dataLogger.ven.vmState=VME_HOLDING_ACK_STATE;
						    break;
						    
						case VME_HOLDING_ACK_STATE:
							key=getKey();
							if(hostAlive<=0)
						    {
							    if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
						  		{	
							  		logVMStats();
							  		logStringRom((const rom unsigned char*)": No VM(s) Running, Going To Sleep.\r\n", DEFAULT_LOG_PF);
								  	flushLog();
					   	 		}  
					   	 		sleepTime(1);
							    //captiveDevice=1;
								cont=0;
								checkKeys=1;
						    }
							else
							if((key==KEY_LONG)||(thisUDL.dataLogger.ven.vmMode & VME_RESTART))
					   	 	{
								restartVMEnvironment(&thisUDL.dataLogger.ven);
								if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
						  		{	
							  	  	if(thisUDL.dataLogger.ven.vmNum>0)
							   		{
							  	 	logF((const rom unsigned char*)": Destroy ", DEFAULT_LOG_PF, (float)thisUDL.dataLogger.ven.vmNum, 0);
							  	 	logStringRom((const rom unsigned char*)" VM(s).\r\n", PF_NOP);
							   		}
							   		logStringRom((const rom unsigned char*)": Restarting Automatically.\r\n", DEFAULT_LOG_PF);
								   	flushLog();
					   	 		} 
   	 						  	setLED(STANDARD_LED_ON_MS);
  					   	 		hostAlive=HOST_ALIVE_ADDITIONAL_TIMEOUT;	
					   	 		cont=0;	
					   	   	}
					   	   	else
					   	   	if(key==KEY_SHORT)
					   	   	{
	   	   	 				  	internalSetLED(3, STANDARD_LED_ON_MS, STANDARD_LED_OFF_MS);   
					   	   	}
						    break;
						}	
						if(checkKeys)doCheckKeys(3);
					}
					else
					{
						cont=0;
					}
		  		}
	  	  	}
	  	  	else
	  	  	{
		  	  		/* In this case, the timeout has expired and there is no memory card, so sleep! */
		  	  		doCheckKeys(3);
		  	  		if(hostAlive<=0)sleepTime(1);
	  	  	}		
    }
}

unsigned char* fillPacket(unsigned char* outptr, unsigned int size, unsigned int* todoSize, unsigned char* inptr, unsigned int* done)
{
	/* copy from inptr to outptr up to size bytes */
	
#if 0
	putrsUART("FP: ");
	disWUART((unsigned int)outptr);
	putrsUART(" SZ: ");
	disWUART((unsigned int)*todoSize);
	putrsUART(" size: ");
	disWUART((unsigned int)size);
#endif
	
	while((size>0)&&((*todoSize)>0))
	{
		*outptr++=*inptr++;
		size--;
		(*todoSize)--;
		(*done)++;
	}
#if 0
	putrsUART(" Done: ");
	disWUART((unsigned int)*done);
	putrsUART("\r\n");
#endif
	return outptr;
}

void initTransfers(void)
{
	transferCounter=0;
	transferPtr=0;
}

void ProcessIOTask(unsigned char usercode)
{   
	TIME_T 			PTtime;
	float 			PTf;
	unsigned char* 	PTptr;
	unsigned int 	PTvID;
	unsigned int	PTvIndex;
	FRESULT 		PTfresult;
	unsigned long   PTLong;
	unsigned char	PTError;
	
	#if USB_ENABLED

		/* Non User code Handlers */    
       
        hostAlive=HOST_ALIVE_TIMEOUT;
    	PTError=0;    
        if(usercode<=1)
        {
       
        switch(OUTPacket.data[0])					//Data arrived, check what kind of command might be in the packet of data.
        { 
	     	default:
	     		PTError++;
	     		break;
	   			  			
	        case CMD_GET_SIZES:		
	        	PTvIndex=4;
	        	PTvID=sizeof(DATA_LOGGER);
	           	INPacket.data[PTvIndex++]=(unsigned char)(PTvID & 0x00FF);
    			INPacket.data[PTvIndex++]=(unsigned char)(PTvID>>8);
    			PTvID=sizeof(VIRTUAL_MACHINE_ENVIRONMENT);
	           	INPacket.data[PTvIndex++]=(unsigned char)(PTvID & 0x00FF);
    			INPacket.data[PTvIndex++]=(unsigned char)(PTvID>>8);
    			PTvID=sizeof(VIRTUAL_MACHINE);
	           	INPacket.data[PTvIndex++]=(unsigned char)(PTvID & 0x00FF);
    			INPacket.data[PTvIndex++]=(unsigned char)(PTvID>>8);
    			PTvID=sizeof(CACHE_OBJECT);
	           	INPacket.data[PTvIndex++]=(unsigned char)(PTvID & 0x00FF);
    			INPacket.data[PTvIndex++]=(unsigned char)(PTvID>>8);
				PTvID=sizeof(CARD_INFO);
	           	INPacket.data[PTvIndex++]=(unsigned char)(PTvID & 0x00FF);
    			INPacket.data[PTvIndex++]=(unsigned char)(PTvID>>8);
				PTvID=NUM_WRITE_SETTINGS;
	           	INPacket.data[PTvIndex++]=(unsigned char)(PTvID & 0x00FF);
    			INPacket.data[PTvIndex++]=(unsigned char)(PTvID>>8);
				break;
	
	case CMD_GET_BOOTLOADER_STATUS:
				INPacket.data[1]=0;
				break;
					
	case CMD_CALL_GLOBAL_FUNCTION_IMMEDIATE:
				argument.arg.uArg=(unsigned short)OUTPacket.data[1];
	        	argument.arg.uArg+=((unsigned short)OUTPacket.data[2]<<8);
	  			PTptr = (unsigned char*)&argument.w;
				*PTptr++=OUTPacket.data[4];
				*PTptr++=OUTPacket.data[5];
				*PTptr++=OUTPacket.data[6];
				*PTptr++=OUTPacket.data[7];
				PTvIndex=8;
				PTvID=0;
				while(PTvID<VM_MAX_ARGUMENTS)
				{
					PTptr=(unsigned char*)&argument.f[PTvID];
					*PTptr++=OUTPacket.data[PTvIndex++];
					*PTptr++=OUTPacket.data[PTvIndex++];
					*PTptr++=OUTPacket.data[PTvIndex++];
					*PTptr++=OUTPacket.data[PTvIndex++];
					PTvID++;
				}
				glCallFunction(0, &argument);
				PTptr=(unsigned char*)&argument.w;
				INPacket.data[0]=0;
				INPacket.data[4]=*PTptr++;
				INPacket.data[5]=*PTptr++;
				INPacket.data[6]=*PTptr++;
				INPacket.data[7]=*PTptr++;
				break;
							
	        case CMD_INIT_TRANSFERS:
	        	PTvID=(unsigned int)OUTPacket.data[1];
	        	PTvID+=((unsigned int)OUTPacket.data[2]<<8);
	        	transferPtr=(unsigned char*)PTvID;
	        	PTvID=(unsigned int)OUTPacket.data[3];
	        	PTvID+=((unsigned int)OUTPacket.data[4]<<8);
	        	transferCounter=(unsigned int)PTvID;
	        	goto transferStatsGoto;
	        	break;
				
	     	case CMD_CONTINUE_BULK_TRANSFER_WRITE:
	     		if(transferCounter>0)
	     		{
		     		PTvID=(unsigned int)OUTPacket.data[1] & 0x00FF;
		     		PTvIndex=0;
		     		fillPacket(transferPtr, PTvID, (unsigned int*)&transferCounter, (unsigned char*)&OUTPacket.data[2], &PTvIndex);	
	     			transferPtr+=(unsigned int)PTvIndex;
	     			if(transferCounter<=0)
	     			{
	     				transferPtr=0;
	     			}	     		
	     		}
	     		else
	     		{
	     			transferPtr=0;
	     			transferCounter=0;
      			}
	    		goto transferStatsGoto;
	        	break;
	     	
	     	case CMD_CONTINUE_BULK_TRANSFER_READ:
	     		if((transferCounter>0)&&(transferPtr!=0))
	     		{
		     		PTvID=(unsigned int)OUTPacket.data[1] & 0x00FF;
		     		PTvIndex=0;
	     			fillPacket(&INPacket.data[2], PTvID, &transferCounter, transferPtr, &PTvIndex);
	     			transferPtr+=(unsigned int)PTvIndex;
	     			if(transferCounter<=0)
    				{
    					transferPtr=0;	
    				}
    				INPacket.data[0]=(unsigned char)(PTvIndex & 0x00FF);
    				INPacket.data[1]=(unsigned char)(PTvIndex>>8);
    			}
    			else
    			{
    				transferPtr=0;
    				transferCounter=0;
				}
				break;  
	
			case CMD_INIT_CARDINFO_TRANSFER:
				transferPtr=(unsigned char*)&cardInfo;
				transferCounter=(unsigned int)sizeof(CARD_INFO);
				goto transferStatsGoto;
	        	break;
				
			case CMD_INIT_DATA_LOGGER_TRANSFER:
				transferPtr=(unsigned char*)&thisUDL.dataLogger;
				transferCounter=(unsigned int)sizeof(DATA_LOGGER);
				goto transferStatsGoto;
	        	break;
	  				
			case CMD_GET_TRANSFER_STATS:
				
				/* We are using a goto to save on code space! */
				transferStatsGoto:	
				INPacket.data[0]=(unsigned char)(transferCounter & 0x00FF);
				INPacket.data[1]=(unsigned char)(transferCounter>>8);
				INPacket.data[2]=(unsigned char)((unsigned int)transferPtr & 0x00FF);
				INPacket.data[3]=(unsigned char)((unsigned int)transferPtr>>8);
				INPacket.data[4]=(unsigned char)(PTvIndex & 0x00FF);
	    		INPacket.data[5]=(unsigned char)(PTvIndex>>8);
				break;
		
			case CMD_SET_CAPTIVE_DEVICE:
				captiveDevice=(unsigned char)OUTPacket.data[1];
				INPacket.data[1]=captiveDevice;
				break;
				
			case CMD_SET_FIRST_POR:
				firstPor=(unsigned char)OUTPacket.data[1];
				INPacket.data[1]=firstPor;
				break;
					
			case CMD_GET_CAPTIVE_DEVICE:
				INPacket.data[0]=0x80;
				INPacket.data[1]=(unsigned char)captiveDevice;
				break;

            case CMD_GET_VERSION:
    			INPacket.data[0]=0x80;
				INPacket.data[1]=0;
				
				PTf=(float)(FIRMWARE_VERSION);
				PTptr = (unsigned char*)&PTf;
				INPacket.data[4]=*PTptr++;
				INPacket.data[5]=*PTptr++;
				INPacket.data[6]=*PTptr++;
				INPacket.data[7]=*PTptr++;
				
				PTptr = (unsigned char*)&devicePORs;
				INPacket.data[8]=*PTptr++;
				INPacket.data[9]=*PTptr++;
				INPacket.data[10]=*PTptr++;
				INPacket.data[11]=*PTptr++;
			    
			    PTf=(float)(SERIAL_BOOTLOADER_VERSION);
				PTptr = (unsigned char*)&PTf;
				INPacket.data[12]=*PTptr++;
				INPacket.data[13]=*PTptr++;
				INPacket.data[14]=*PTptr++;
				INPacket.data[15]=*PTptr++;
			
				PTLong=(unsigned long)BOOTLOADER_VERSION;
				PTptr=(unsigned char*)&PTLong;
				INPacket.data[16]=*PTptr++;
				INPacket.data[17]=*PTptr++;
				INPacket.data[18]=*PTptr++;
				INPacket.data[19]=*PTptr++;
			    break;
			
			case CMD_RESTORE_DEFAULTS:
				#if(USE_EEPROM)
				
				#else
					DataEEDeleteFile();
				#endif
				DataEEWrite(EE_MAGIC_VALUE+1, EE_MAGIC_ADDR);
			case CMD_POR_RESET:
				RCONbits.POR=0;
			case CMD_RESET:
				_asm 
					reset
				_endasm
				break;
				
			case CMD_SET_TIME:
				PTptr=(unsigned char*)&PTtime;
				for(PTvID=0; PTvID<sizeof(PTtime); PTvID++)
				{
					*PTptr++=OUTPacket.data[4+PTvID];
				}
				setTime(&PTtime);
				break;
	       
	       	case CMD_GET_TIME:
	       		getTime(&PTtime);
				PTptr=(unsigned char*)&PTtime;
				for(PTvID=0; PTvID<sizeof(PTtime); PTvID++)
				{
					INPacket.data[4+PTvID]=*PTptr++;
				}
				break;
			
	    }
	   
	   }
	   
	   if(usercode==1)
	   {
	   
	    switch(OUTPacket.data[0])					//Data arrived, check what kind of command might be in the packet of data.
        { 
	     	default:
	     		PTError++;
	     		break;
	     		
			case CMD_ENTER_USB_BOOTLOADER:
	  			bootFunction=1;
	  			enterBootLoader=1;
	  			break;
	  			
	  		case CMD_ENTER_SERIAL_BOOTLOADER:
	  			bootFunction=0;
	  			enterBootLoader=1;
	  			break;
	     			
	     	case CMD_LOGSTRING:
				PTvID=(unsigned int)OUTPacket.data[1];
	        	PTvID+=((unsigned int)OUTPacket.data[2]<<8);
	        	logString((unsigned char*)&OUTPacket.data[4], PTvID);
	        	break;
	        
	    	case CMD_GET_SERIAL_USB_BUFFER:
                flushSerialUSB();
                break;
			
			case CMD_INIT_SERIAL_USB_BUFFER:
				initSerialUSB();
				break;
		   	
			case CMD_FLUSH_LOG:
				flushLog();
				break;
			
			case CMD_DELETE_LOG:
				deleteLog();
				break;
			
			case CMD_RESTART_LOG:
				restartLog();
				break;
				
			case CMD_FLUSH_SETTINGS:
				flushSettings();
				break;
							   
	        case CMD_RECONNECT_MEMORYCARD:
	        		reconnectMemoryCard();
	        		break;
	        		        
	        case CMD_UPDATE_CARD_INFO:
	           	   	updateFreeSize(&cardInfo);
	           	   	break;
	           	   		
	    	case CMD_OPEN_READ_FILE:
				/* open a file for reading, the path is given beginning from OUTPacket.data[4] */
				cardLED=CARD_LED_TIMEOUT;  
        		fileFull=0;
        		fileIndex=0;
        		f_close(&fsrc);
				PTfresult=f_open(&fsrc, (char*) &OUTPacket.data[4], FA_OPEN_EXISTING | FA_READ);
				INPacket.data[0]=(unsigned char)PTfresult;						
		   	 	fastTasks=FAST_TASKS_SECS;
				break;

			case CMD_UNLINK_FILE:
				/* Delete a File */
				PTfresult=f_unlink((char*)&OUTPacket.data[4]);
				INPacket.data[0]=(unsigned char)PTfresult;						
		   	 	fastTasks=FAST_TASKS_SECS;
				break;
			
			case CMD_MKDIR:
				/* Create a Directory */
				PTfresult=f_mkdir((char*)&OUTPacket.data[4]);
				INPacket.data[0]=(unsigned char)PTfresult;
		   	 	fastTasks=FAST_TASKS_SECS;
				break;
							
			case CMD_OPEN_DIR:
				/* open a directory the path is given beginning from OUTPacket.data[4] */
				PTfresult=f_opendir(&fdir, (char*)&OUTPacket.data[4]);
				INPacket.data[0]=(unsigned char)PTfresult;						
		   	 	fastTasks=FAST_TASKS_SECS;
				break;
			
			case CMD_READ_DIR:
				/* read a directory item */			
				finfo.lfname=(WCHAR*)&INPacket.fdir.fname;
				finfo.lfsize=(USB_FDIR_NAME_SIZE);
				*finfo.lfname=0;
				PTfresult=f_readdir(&fdir, &finfo);
				INPacket.fdir.fresult=PTfresult;
				INPacket.fdir.fdate=finfo.fdate;
				INPacket.fdir.ftime=finfo.ftime;
				INPacket.fdir.fattrib=finfo.fattrib;
				INPacket.fdir.fsize=finfo.fsize;
				if(*finfo.lfname)
				{
					INPacket.fdir.lfsize=finfo.lfsize;
				} 
				else 
				{
					mem_cpy((unsigned char*)INPacket.fdir.fname, (unsigned char*)finfo.fname, 13);
					INPacket.fdir.lfsize=13;
				}				
		   	 	fastTasks=FAST_TASKS_SECS;
				break;
				
			case CMD_READ_DIR_REWIND:
				/* rewind directory list by providing null pointer */			
				PTfresult=f_readdir(&fdir, 0);
				INPacket.fdir.fresult=PTfresult;
		   	 	fastTasks=FAST_TASKS_SECS;
				break;
				
			case CMD_OPEN_WRITE_FILE:
				fileIndex=0;
				fileFull=0;
				/* open a file for writing, the path is given beginning from OUTPacket.data[4] */
				cardLED=CARD_LED_TIMEOUT;  
        		f_close(&fsrc);
				PTfresult=f_open(&fsrc, (char*)&OUTPacket.data[4], FA_CREATE_ALWAYS | FA_WRITE);
				INPacket.data[0]=(unsigned char)PTfresult;
		   	 	fastTasks=FAST_TASKS_SECS;
				break;
				
			case CMD_READ_FILE:
				/* 
					read up to 60 bytes from a previously opened file... 
				*/
				/* get the length... */
				cardLED=CARD_LED_TIMEOUT;
				PTvID=(unsigned int)OUTPacket.data[2];
				PTvID+=((unsigned int)OUTPacket.data[3]<<8);
				if(PTvID>60)PTvID=60;
	       		PTfresult=f_read(&fsrc, (unsigned char*)&INPacket.data[4], PTvID, (unsigned int*)&PTvIndex);
	     		if(PTfresult!=FR_OK)PTvIndex=0;
	     		INPacket.data[0]=(unsigned char)PTfresult;
	     		INPacket.data[2]=(unsigned char)PTvIndex;
	     		INPacket.data[3]=(unsigned char)(PTvIndex>>8);	     		
	       	 	fastTasks=FAST_TASKS_SECS;
		 		break;
					
		 	case CMD_WRITE_FILE:
				/* 
					write up to 60 bytes from a previously opened file... 
				*/
				
				/* get the length... */
				cardLED=CARD_LED_TIMEOUT;
				PTvID=(unsigned int)OUTPacket.data[2];
				PTvID+=((unsigned int)OUTPacket.data[3]<<8);
				if(PTvID>60)PTvID=60;
	   			PTfresult=f_write(&fsrc, (unsigned char*)&OUTPacket.data[4], PTvID, (unsigned int*)&PTvIndex);
	     		if(PTfresult!=FR_OK)PTvIndex=0;	     		
	     		INPacket.data[0]=(unsigned char)PTfresult;
	     		INPacket.data[2]=(unsigned char)PTvIndex;
	     		INPacket.data[3]=(unsigned char)(PTvIndex>>8);
	       	 	fastTasks=FAST_TASKS_SECS;
		 		break;
				
			case CMD_CLOSE_FILE:
				PTfresult=f_close(&fsrc);
				INPacket.data[0]=PTfresult;
		   	 	fastTasks=FAST_TASKS_SECS;
				break;	 			
	    }
	    
	   }
	 
	   if(PTError>usercode)
	   {
	   
	   
	   }
	   else
	   if(!USBHandleBusy(USBGenericInHandle))		
	   {	
			USBGenericInHandle = USBGenWrite(USBGEN_EP_NUM,(BYTE*)&INPacket.data[0],USBGEN_EP_SIZE);	
       }
				
	 //Re-arm the OUT endpoint for the next packet:
     //The USBGenRead() function call "arms" the endpoint (and makes it "busy").  If the endpoint is armed, the SIE will 
     //automatically accept data from the host, if the host tries to send a packet of data to the endpoint.  Once a data 
     //packet addressed to this endpoint is received from the host, the endpoint will no longer be busy, and the application
     //can read the data which will be sitting in the buffer.
 	  USBGenericOutHandle = USBGenRead(USBGEN_EP_NUM,(BYTE*)&OUTPacket.data,USBGEN_EP_SIZE);   
#endif
}//end ProcessIO



